<?php 
session_start();
if (!isset($_SESSION['user'])) {
    header("Location: login.php");
    exit;
}
$loggedInUser = $_SESSION['user'];
require_once 'dbconfig.php'; 
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>OLT Port Management</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://unpkg.com/leaflet@1.9.4/dist/leaflet.css" />
    <link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet" />
    <style>
        /* Custom CSS for vertical slots and horizontal ports */
        .slot-container {
            display: flex;
            flex-direction: column;
            gap: 10px;
        }
        
        .slot-row {
            display: flex;
            gap: 10px;
            align-items: flex-start;
        }
        
        .slot-header {
            background-color: #f8f9fa;
            padding: 5px 10px;
            border-radius: 4px;
            font-weight: bold;
            text-align: center;
            margin-bottom: 5px;
        }
        
        .port-container {
            display: flex;
            flex-wrap: wrap;
            gap: 5px;
            padding: 5px;
            background-color: #f1f1f1;
            border-radius: 4px;
        }
        
        .port-item {
            display: flex;
            flex-direction: column;
            align-items: center;
            cursor: pointer;
            padding: 5px;
            border-radius: 4px;
            transition: all 0.2s;
        }
        
        .port-item:hover {
            background-color: #e9ecef;
        }
        
        .port-circle {
            width: 30px;
            height: 30px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            background-color: #fff;
            border: 1px solid #dee2e6;
            margin-bottom: 3px;
            font-size: 12px;
        }
        
        .port-used .port-circle {
            background-color: #d1e7dd;
            border-color: #badbcc;
        }
        
        .port-unused .port-circle {
            background-color: #f8d7da;
            border-color: #f5c2c7;
        }
        
        .fdt-badge {
            font-size: 10px;
            padding: 2px 5px;
            border-radius: 3px;
            color: white;
            text-align: center;
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
            max-width: 80px;
        }
        
        .fdt-badge.RFS {
            background-color: #198754;
        }
        
        .fdt-badge.Construction {
            background-color: #fd7e14;
        }
        
        .fdt-badge.Design {
            background-color: #0dcaf0;
        }
        
        .fdt-badge.Unused {
            background-color: #6c757d;
        }
        
        .legend-item {
            display: flex;
            align-items: center;
            gap: 5px;
            font-size: 12px;
        }
        
        .legend-badge {
            width: 15px;
            height: 15px;
            border-radius: 3px;
            display: inline-block;
        }
        
        .map-marker {
            background-color: #0d6efd;
            width: 24px;
            height: 24px;
            border-radius: 50%;
            border: 2px solid white;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: bold;
        }
        
        .select2-container {
            width: 100% !important;
        }
    </style>
</head>
<body>
<div class="container-fluid py-3">
    <div class="row mb-3">
        <div class="col-md-6">
            <h5>OLT Port Management</h5>
        </div>
        <div class="col-md-6 text-end">
            <button type="button" class="btn btn-success" data-bs-toggle="modal" data-bs-target="#addOltModal">
                Add New OLT
            </button>
        </div>
    </div>

    <div class="row mb-3">
        <div class="col-md-3">
            <select class="form-select select2" id="siteSelect">
                <option value="">Select OLT Site</option>
                <?php
                $db = getDBConnection();
                $sites = $db->query("SELECT DISTINCT p.siteid, COALESCE(a.status, 'Unused') as status 
                                  FROM ports p 
                                  LEFT JOIN assetdata a ON p.siteid = a.siteid 
                                  ORDER BY p.siteid");
                while ($site = $sites->fetch_assoc()) {
                    echo '<option value="'.htmlspecialchars($site['siteid']).'" data-status="'.htmlspecialchars($site['status']).'">'
                        .htmlspecialchars($site['siteid']).'</option>';
                }
                $db->close();
                ?>
            </select>
        </div>
        <div class="col-md-2">
            <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addSlotModal">
                Add New Slot
            </button>
        </div>
    </div>

    <div class="card mb-3" id="oltContainer" style="display:none;">
        <div class="card-body p-3">
            <h6 id="oltTitle" class="mb-3"></h6>
            <div class="mb-3 d-flex flex-wrap align-items-center gap-3" id="legendStatus">
                <div class="legend-item">
                    <span class="legend-badge fdt-badge RFS"></span> RFS
                </div>
                <div class="legend-item">
                    <span class="legend-badge fdt-badge Construction"></span> Construction
                </div>
                <div class="legend-item">
                    <span class="legend-badge fdt-badge Design"></span> Design
                </div>
                <div class="legend-item">
                    <span class="legend-badge fdt-badge Unused"></span> Available
                </div>
            </div>
            <div class="slot-container" id="slotContainer"></div>
        </div>
    </div>
    
    <div class="card mb-3" id="mapContainer" style="display:none;">
        <div class="card-header">
            <h6 class="mb-0">Lokasi OLT</h6>
        </div>
        <div class="card-body p-0">
            <div id="oltMap" style="height: 300px;"></div>
        </div>
    </div>
</div>

<!-- Modal Add OLT -->
<div class="modal fade" id="addOltModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title">Add New OLT</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>
      <div class="modal-body">
        <form id="addOltForm">
            <div class="mb-3">
                <label class="form-label">Site ID</label>
                <input type="text" class="form-control" id="newSiteId" required>
            </div>
        </form>
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
        <button type="button" class="btn btn-primary" id="confirmAddOlt">Save</button>
      </div>
    </div>
  </div>
</div>

<!-- Modal Add Slot -->
<div class="modal fade" id="addSlotModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title">Add New Slot</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>
      <div class="modal-body">
        <form id="addSlotForm">
            <div class="mb-3">
                <label class="form-label">Site ID</label>
                <select class="form-select select2" id="addSiteId" required>
                    <option value="">Select Site</option>
                    <?php
                    $db = getDBConnection();
                    $sites = $db->query("SELECT DISTINCT p.siteid, COALESCE(a.status, 'Unused') as status 
                                      FROM ports p 
                                      LEFT JOIN assetdata a ON p.siteid = a.siteid 
                                      ORDER BY p.siteid");
                    while ($site = $sites->fetch_assoc()) {
                        echo '<option value="'.htmlspecialchars($site['siteid']).'">'.htmlspecialchars($site['siteid']).'</option>';
                    }
                    $db->close();
                    ?>
                </select>
            </div>
            <div class="mb-3">
                <label class="form-label">Slot Number</label>
                <input type="number" class="form-control" id="addSlotId" min="0" max="14" required>
            </div>
        </form>
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
        <button type="button" class="btn btn-primary" id="confirmAddSlot">Save</button>
      </div>
    </div>
  </div>
</div>

<!-- Modal Edit Port -->
<div class="modal fade" id="portModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title" id="modalTitle">Port Details</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>
      <div class="modal-body">
        <form id="portForm">
            <input type="hidden" id="modalSiteId">
            <input type="hidden" id="modalSlotId">
            <input type="hidden" id="modalPortNum">
            <div class="mb-3">
                <label class="form-label">FDT</label>
                <input type="text" class="form-control" id="modalFdtCode" required>
            </div>
            <div class="mb-3">
                <label class="form-label">Updated By</label>
               <input type="text" class="form-control" id="modalUpdateBy" value="<?= htmlspecialchars($loggedInUser) ?>" readonly>
            </div>
        </form>
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-danger me-auto" id="deletePortBtn">Delete</button>
        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
        <button type="button" class="btn btn-primary" id="savePortBtn">Save</button>
      </div>
    </div>
  </div>
</div>

<script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
<script src="https://unpkg.com/leaflet@1.9.4/dist/leaflet.js"></script>
<script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>

<script>
    // Global variables
    let currentSite = '';
    let currentSlot = 0;
    let currentPort = 0;
    let numberingBase = 1; // Default to 1-based numbering
    const portModal = new bootstrap.Modal('#portModal');
    const addSlotModal = new bootstrap.Modal('#addSlotModal');
    const addOltModal = new bootstrap.Modal('#addOltModal');
    let oltMap = null;
    let mapMarker = null;
    
    // Initialize Select2 dropdowns
    $(document).ready(function() {
        $('#siteSelect').select2({
            placeholder: "Select OLT Site",
            allowClear: true,
            width: '100%'
        });
        
        $('#addSiteId').select2({
            placeholder: "Select Site",
            allowClear: true,
            width: '100%',
            dropdownParent: $('#addSlotModal')
        });
    });
    
    // Initialize event listeners
    $('#siteSelect').on('change', loadOLT);
    document.getElementById('confirmAddSlot').addEventListener('click', addNewSlot);
    document.getElementById('confirmAddOlt').addEventListener('click', addNewOlt);
    document.getElementById('savePortBtn').addEventListener('click', savePort);
    document.getElementById('deletePortBtn').addEventListener('click', deletePort);
    
    // Load OLT data
    async function loadOLT() {
        currentSite = this.value;
        const selectedOption = $(this).find('option:selected');
        const siteStatus = selectedOption.data('status') || 'Unused';
        
        if (!currentSite) {
            document.getElementById('oltContainer').style.display = 'none';
            document.getElementById('mapContainer').style.display = 'none';
            if (oltMap) {
                oltMap.remove();
                oltMap = null;
            }
            return;
        }
        
        // Update title with site status badge
        document.getElementById('oltTitle').innerHTML = `
            OLT:  <span class="fdt-badge ${siteStatus}">${currentSite}</span>
        `;
        document.getElementById('oltContainer').style.display = 'block';
        
        try {
            const [slotsResponse, locationResponse] = await Promise.all([
                fetch(`get_slots.php?siteid=${encodeURIComponent(currentSite)}`),
                fetch(`get_location.php?siteid=${encodeURIComponent(currentSite)}`)
            ]);
            
            const slotsData = await slotsResponse.json();
            const locationData = await locationResponse.json();
            
            renderSlots(slotsData);
            showMap(locationData);
        } catch (error) {
            console.error('Error loading OLT:', error);
        }
    }
    
    function showMap(locationData) {
        document.getElementById('mapContainer').style.display = 'block';
        
        // Initialize map if not exists
        if (!oltMap) {
            oltMap = L.map('oltMap').setView([-2.5489, 118.0149], 5); // Default view Indonesia
            
            L.tileLayer('http://{s}.basemaps.cartocdn.com/light_all/{z}/{x}/{y}.png', {
                attribution: '&copy; <a href="https://www.openstreetmap.org/copyright">OpenStreetMap</a> contributors'
            }).addTo(oltMap);
        }
        
        // If location data exists
        if (locationData.lat && locationData.lng) {
            const latLng = [parseFloat(locationData.lat), parseFloat(locationData.lng)];
            
            // Set view to OLT location
            oltMap.setView(latLng, 15);
            
            // Remove old marker if exists
            if (mapMarker) {
                oltMap.removeLayer(mapMarker);
            }
            
            // Add new marker
            mapMarker = L.marker(latLng, {
                icon: L.divIcon({
                    className: 'map-marker',
                    iconSize: [24, 24]
                })
            }).addTo(oltMap);
            
            // Add popup
            mapMarker.bindPopup(`<b>${currentSite}</b><br>Lat: ${latLng[0]}<br>Lng: ${latLng[1]}`).openPopup();
        } else {
            // If no location data
            alert('Lokasi OLT tidak ditemukan dalam database');
        }
    }

    // Render slots and ports in vertical layout with horizontal ports
    function renderSlots(slotsData) {
        const slotContainer = document.getElementById('slotContainer');
        slotContainer.innerHTML = '';

        const slotStart = numberingBase;
        const slotEnd = numberingBase + 15;
        const portStart = numberingBase;
        const portEnd = numberingBase + 15;

        for (let slotNum = slotStart; slotNum <= slotEnd; slotNum++) {
            const slotRow = document.createElement('div');
            slotRow.className = 'slot-row';
            
            const slotHeader = document.createElement('div');
            slotHeader.className = 'slot-header';
            slotHeader.textContent = `Slot ${slotNum}`;
            
            const portContainer = document.createElement('div');
            portContainer.className = 'port-container';
            
            const dbSlotNum = numberingBase === 1 ? slotNum : slotNum - 1;
            const slotData = slotsData[dbSlotNum] || {};

            for (let portNum = portStart; portNum <= portEnd; portNum++) {
                const dbPortNum = numberingBase === 0 ? portNum : portNum - 1;
                const portCol = 'port' + (dbPortNum + 1);
                const statusKey = 'status' + (dbPortNum + 1);

                const portValue = slotData[portCol] || 'Unused';
                const isUsed = portValue !== 'Unused';
                const status = slotData[statusKey] || 'Unused';

                const portDiv = document.createElement('div');
                portDiv.className = `port-item ${isUsed ? 'port-used' : 'port-unused'}`;
                portDiv.innerHTML = `
                    <div class="port-circle">${portNum}</div>
                    <div class="fdt-badge ${status}">${portValue}</div>
                `;

                portDiv.addEventListener('click', () => {
                    selectPort(dbSlotNum, dbPortNum, portNum, slotNum);
                });

                portContainer.appendChild(portDiv);
            }

            slotRow.appendChild(slotHeader);
            slotRow.appendChild(portContainer);
            slotContainer.appendChild(slotRow);
        }
    }

    // Select port handler
    async function selectPort(dbSlotNum, dbPortNum, displayPortNum, displaySlotNum) {
        currentSlot = dbSlotNum;
        currentPort = dbPortNum;
        
        try {
            const response = await fetch(`get_port.php?siteid=${currentSite}&slotid=${dbSlotNum}&port=${dbPortNum}`);
            const portData = await response.json();
            
            document.getElementById('modalTitle').textContent = 
                `Port ${displayPortNum} - Slot ${displaySlotNum}`;
            document.getElementById('modalSiteId').value = currentSite;
            document.getElementById('modalSlotId').value = dbSlotNum;
            document.getElementById('modalPortNum').value = dbPortNum;
            document.getElementById('modalFdtCode').value = portData.fdt_code || '';
            
            portModal.show();
        } catch (error) {
            console.error('Error loading port:', error);
        }
    }
    
    // Add new slot
    async function addNewSlot() {
        const siteid = document.getElementById('addSiteId').value;
        const displaySlotNum = parseInt(document.getElementById('addSlotId').value);
        
        if (!siteid || isNaN(displaySlotNum)) {
            alert('Please fill all fields correctly');
            return;
        }
        
        const dbSlotNum = numberingBase === 0 ? displaySlotNum : displaySlotNum - 1;
        
        try {
            const response = await fetch('add_slot.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ 
                    siteid, 
                    slotid: dbSlotNum,
                    numberingBase
                })
            });
            
            const result = await response.json();
            
            if (result.success) {
                addSlotModal.hide();
                document.getElementById('addSlotForm').reset();
                if (currentSite === siteid) {
                    loadOLT.call({ value: siteid });
                }
            } else {
                alert(result.error || 'Failed to add slot');
            }
        } catch (error) {
            console.error('Error adding slot:', error);
        }
    }
    
    // Add new OLT
    async function addNewOlt() {
        const siteid = document.getElementById('newSiteId').value.trim();
        
        if (!siteid) {
            alert('Please enter a Site ID');
            return;
        }
        
        try {
            const response = await fetch('add_olt.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ siteid: siteid })
            });
            
            const result = await response.json();
            
            if (result.success) {
                const option = document.createElement('option');
                option.value = siteid;
                option.textContent = siteid;
                option.setAttribute('data-status', 'Unused');
                
                document.getElementById('siteSelect').appendChild(option);
                
                const addSlotOption = option.cloneNode(true);
                document.getElementById('addSiteId').appendChild(addSlotOption);
                
                // Refresh Select2 dropdowns
                $('#siteSelect').trigger('change');
                $('#addSiteId').trigger('change');
                
                addOltModal.hide();
                document.getElementById('addOltForm').reset();
                alert('OLT added successfully!');
            } else {
                alert(result.error || 'Failed to add OLT');
            }
        } catch (error) {
            console.error('Error adding OLT:', error);
            alert('An error occurred while adding the OLT');
        }
    }
    
    // Save port changes
    async function savePort() {
        const portData = {
            siteid: document.getElementById('modalSiteId').value,
            slotid: parseInt(document.getElementById('modalSlotId').value),
            port: parseInt(document.getElementById('modalPortNum').value),
            fdt_code: document.getElementById('modalFdtCode').value,
            update_by: document.getElementById('modalUpdateBy').value,
            numberingBase
        };
        
        try {
            const response = await fetch('update_port.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(portData)
            });
            
            const result = await response.json();
            
            if (result.success) {
                portModal.hide();
                loadOLT.call({ value: currentSite });
            } else {
                alert(result.error || 'Failed to save port');
            }
        } catch (error) {
            console.error('Error saving port:', error);
        }
    }
    
    // Delete port
    async function deletePort() {
        if (!confirm('Are you sure you want to delete this port?')) return;
        
        const portData = {
            siteid: document.getElementById('modalSiteId').value,
            slotid: parseInt(document.getElementById('modalSlotId').value),
            port: parseInt(document.getElementById('modalPortNum').value),
            numberingBase
        };
        
        try {
            const response = await fetch('delete_port.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(portData)
            });
            
            const result = await response.json();
            
            if (result.success) {
                portModal.hide();
                loadOLT.call({ value: currentSite });
            } else {
                alert(result.error || 'Failed to delete port');
            }
        } catch (error) {
            console.error('Error deleting port:', error);
        }
    }
</script>
</body>
</html>