<?php 
session_start();
if (!isset($_SESSION['user'])) {
    header("Location: login.php");
    exit;
}
$loggedInUser = $_SESSION['user'];
require_once 'dbconfig.php'; 
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>OLT Port Management</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="gaya_olt.css" rel="stylesheet">
</head>
<body>
<div class="container-fluid py-3">
    <div class="row mb-3">
        <div class="col-md-6">
            <h5>OLT Port Management</h5>
        </div>
        <div class="col-md-6 text-end">
            <button type="button" class="btn btn-success" data-bs-toggle="modal" data-bs-target="#addOltModal">
                Add New OLT
            </button>
        </div>
    </div>

    <div class="row mb-3">
        <div class="col-md-3">
            <select class="form-select" id="siteSelect">
                <option value="">Select OLT Site</option>
                <?php
                $db = getDBConnection();
                $sites = $db->query("SELECT DISTINCT p.siteid, COALESCE(a.status, 'Unused') as status 
                                  FROM ports p 
                                  LEFT JOIN assetdata a ON p.siteid = a.siteid 
                                  ORDER BY p.siteid");
                while ($site = $sites->fetch_assoc()) {
                    echo '<option value="'.htmlspecialchars($site['siteid']).'" data-status="'.htmlspecialchars($site['status']).'">'
                        .htmlspecialchars($site['siteid']).'</option>';
                }
                $db->close();
                ?>
            </select>
        </div>
        <div class="col-md-2">
            <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addSlotModal">
                Add New Slot
            </button>
        </div>
    </div>

    <div class="card mb-3" id="oltContainer" style="display:none;">
        <div class="card-body p-3">
            <h6 id="oltTitle" class="mb-3"></h6>
            <div class="mb-3 d-flex flex-wrap align-items-center gap-3" id="legendStatus">
                <div class="legend-item">
                    <span class="legend-badge fdt-badge RFS"></span> RFS
                </div>
                <div class="legend-item">
                    <span class="legend-badge fdt-badge Construction"></span> Construction
                </div>
                <div class="legend-item">
                    <span class="legend-badge fdt-badge Design"></span> Design
                </div>
                <div class="legend-item">
                    <span class="legend-badge fdt-badge Unused"></span> Available
                </div>
            </div>
            <div class="slot-container" id="slotContainer"></div>
        </div>
    </div>
</div>

<!-- Modal Add OLT -->
<div class="modal fade" id="addOltModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title">Add New OLT</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>
      <div class="modal-body">
        <form id="addOltForm">
            <div class="mb-3">
                <label class="form-label">Site ID</label>
                <input type="text" class="form-control" id="newSiteId" required>
            </div>
        </form>
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
        <button type="button" class="btn btn-primary" id="confirmAddOlt">Save</button>
      </div>
    </div>
  </div>
</div>

<!-- Modal Add Slot -->
<div class="modal fade" id="addSlotModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title">Add New Slot</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>
      <div class="modal-body">
        <form id="addSlotForm">
            <div class="mb-3">
                <label class="form-label">Site ID</label>
                <select class="form-select" id="addSiteId" required>
                    <option value="">Select Site</option>
                    <?php
                    $db = getDBConnection();
                    $sites = $db->query("SELECT DISTINCT p.siteid, COALESCE(a.status, 'Unused') as status 
                                      FROM ports p 
                                      LEFT JOIN assetdata a ON p.siteid = a.siteid 
                                      ORDER BY p.siteid");
                    while ($site = $sites->fetch_assoc()) {
                        echo '<option value="'.htmlspecialchars($site['siteid']).'">'.htmlspecialchars($site['siteid']).'</option>';
                    }
                    $db->close();
                    ?>
                </select>
            </div>
            <div class="mb-3">
                <label class="form-label">Slot Number</label>
                <input type="number" class="form-control" id="addSlotId" min="0" max="14" required>
            </div>
        </form>
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
        <button type="button" class="btn btn-primary" id="confirmAddSlot">Save</button>
      </div>
    </div>
  </div>
</div>

<!-- Modal Edit Port -->
<div class="modal fade" id="portModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog">
    <div class="modal-content">
      <div class="modal-header">
        <h5 class="modal-title" id="modalTitle">Port Details</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>
      <div class="modal-body">
        <form id="portForm">
            <input type="hidden" id="modalSiteId">
            <input type="hidden" id="modalSlotId">
            <input type="hidden" id="modalPortNum">
            <div class="mb-3">
                <label class="form-label">FDT/ODP Code</label>
                <input type="text" class="form-control" id="modalFdtCode" required>
            </div>
            <div class="mb-3">
                <label class="form-label">Updated By</label>
               <input type="text" class="form-control" id="modalUpdateBy" value="<?= htmlspecialchars($loggedInUser) ?>" readonly>
            </div>
        </form>
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-danger me-auto" id="deletePortBtn">Delete</button>
        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
        <button type="button" class="btn btn-primary" id="savePortBtn">Save</button>
      </div>
    </div>
  </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>

<script>
    // Global variables
    let currentSite = '';
    let currentSlot = 0;
    let currentPort = 0;
    let numberingBase = 1; // Default to 1-based numbering
    const portModal = new bootstrap.Modal('#portModal');
    const addSlotModal = new bootstrap.Modal('#addSlotModal');
    const addOltModal = new bootstrap.Modal('#addOltModal');
    
    // Initialize
    document.getElementById('siteSelect').addEventListener('change', loadOLT);
    document.getElementById('confirmAddSlot').addEventListener('click', addNewSlot);
    document.getElementById('confirmAddOlt').addEventListener('click', addNewOlt);
    document.getElementById('savePortBtn').addEventListener('click', savePort);
    document.getElementById('deletePortBtn').addEventListener('click', deletePort);
    
    // Load OLT data
    async function loadOLT() {
        currentSite = this.value;
        const selectedOption = this.options[this.selectedIndex];
        const siteStatus = selectedOption.getAttribute('data-status') || 'Unused';
        
        if (!currentSite) {
            document.getElementById('oltContainer').style.display = 'none';
            return;
        }
        
        // Update title with site status badge
        document.getElementById('oltTitle').innerHTML = `
            OLT: ${currentSite}
            <span class="fdt-badge ${siteStatus}">${siteStatus}</span>
        `;
        document.getElementById('oltContainer').style.display = 'block';
        
        try {
            const response = await fetch(`get_slots.php?siteid=${encodeURIComponent(currentSite)}`);
            const slotsData = await response.json();
            renderSlots(slotsData);
        } catch (error) {
            console.error('Error loading OLT:', error);
        }
    }

    // Render slots and ports
    function renderSlots(slotsData) {
        const slotContainer = document.getElementById('slotContainer');
        slotContainer.innerHTML = '';

        const slotStart = numberingBase;
        const slotEnd = numberingBase + 15;
        const portStart = numberingBase;
        const portEnd = numberingBase + 15;

        for (let slotNum = slotStart; slotNum <= slotEnd; slotNum++) {
            const slotDiv = document.createElement('div');
            slotDiv.className = 'slot-column';
            slotDiv.innerHTML = `<div class="slot-header">Slot ${slotNum}</div>`;

            const dbSlotNum = numberingBase === 1 ? slotNum : slotNum - 1;
            const slotData = slotsData[dbSlotNum] || {};

            for (let portNum = portStart; portNum <= portEnd; portNum++) {
                const dbPortNum = numberingBase === 0 ? portNum : portNum - 1;
                const portCol = 'port' + (dbPortNum + 1);
                const statusKey = 'status' + (dbPortNum + 1);

                const portValue = slotData[portCol] || 'Unused';
                const isUsed = portValue !== 'Unused';
                const status = slotData[statusKey] || 'Unused';

                const portDiv = document.createElement('div');
                portDiv.className = `port-item ${isUsed ? 'port-used' : 'port-unused'}`;
                portDiv.innerHTML = `
                    <div class="port-circle">${portNum}</div>
                    <div class="fdt-badge ${status}">${portValue}</div>
                `;

                portDiv.addEventListener('click', () => {
                    selectPort(dbSlotNum, dbPortNum, portNum, slotNum);
                });

                slotDiv.appendChild(portDiv);
            }

            slotContainer.appendChild(slotDiv);
        }
    }

    // Select port handler
    async function selectPort(dbSlotNum, dbPortNum, displayPortNum, displaySlotNum) {
        currentSlot = dbSlotNum;
        currentPort = dbPortNum;
        
        try {
            const response = await fetch(`get_port.php?siteid=${currentSite}&slotid=${dbSlotNum}&port=${dbPortNum}`);
            const portData = await response.json();
            
            document.getElementById('modalTitle').textContent = 
                `Port ${displayPortNum} - Slot ${displaySlotNum}`;
            document.getElementById('modalSiteId').value = currentSite;
            document.getElementById('modalSlotId').value = dbSlotNum;
            document.getElementById('modalPortNum').value = dbPortNum;
            document.getElementById('modalFdtCode').value = portData.fdt_code || '';
            
            portModal.show();
        } catch (error) {
            console.error('Error loading port:', error);
        }
    }
    
    // Add new slot
    async function addNewSlot() {
        const siteid = document.getElementById('addSiteId').value;
        const displaySlotNum = parseInt(document.getElementById('addSlotId').value);
        
        if (!siteid || isNaN(displaySlotNum)) {
            alert('Please fill all fields correctly');
            return;
        }
        
        const dbSlotNum = numberingBase === 0 ? displaySlotNum : displaySlotNum - 1;
        
        try {
            const response = await fetch('add_slot.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ 
                    siteid, 
                    slotid: dbSlotNum,
                    numberingBase
                })
            });
            
            const result = await response.json();
            
            if (result.success) {
                addSlotModal.hide();
                document.getElementById('addSlotForm').reset();
                if (currentSite === siteid) {
                    loadOLT.call({ value: siteid });
                }
            } else {
                alert(result.error || 'Failed to add slot');
            }
        } catch (error) {
            console.error('Error adding slot:', error);
        }
    }
    
    // Add new OLT
    async function addNewOlt() {
        const siteid = document.getElementById('newSiteId').value.trim();
        
        if (!siteid) {
            alert('Please enter a Site ID');
            return;
        }
        
        try {
            const response = await fetch('add_olt.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ siteid: siteid })
            });
            
            const result = await response.json();
            
            if (result.success) {
                const option = document.createElement('option');
                option.value = siteid;
                option.textContent = siteid;
                
                document.getElementById('siteSelect').appendChild(option);
                
                const addSlotOption = option.cloneNode(true);
                document.getElementById('addSiteId').appendChild(addSlotOption);
                
                addOltModal.hide();
                document.getElementById('addOltForm').reset();
                alert('OLT added successfully!');
            } else {
                alert(result.error || 'Failed to add OLT');
            }
        } catch (error) {
            console.error('Error adding OLT:', error);
            alert('An error occurred while adding the OLT');
        }
    }
    
    // Save port changes
    async function savePort() {
        const portData = {
            siteid: document.getElementById('modalSiteId').value,
            slotid: parseInt(document.getElementById('modalSlotId').value),
            port: parseInt(document.getElementById('modalPortNum').value),
            fdt_code: document.getElementById('modalFdtCode').value,
            update_by: document.getElementById('modalUpdateBy').value,
            numberingBase
        };
        
        try {
            const response = await fetch('update_port.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(portData)
            });
            
            const result = await response.json();
            
            if (result.success) {
                portModal.hide();
                loadOLT.call({ value: currentSite });
            } else {
                alert(result.error || 'Failed to save port');
            }
        } catch (error) {
            console.error('Error saving port:', error);
        }
    }
    
    // Delete port
    async function deletePort() {
        if (!confirm('Are you sure you want to delete this port?')) return;
        
        const portData = {
            siteid: document.getElementById('modalSiteId').value,
            slotid: parseInt(document.getElementById('modalSlotId').value),
            port: parseInt(document.getElementById('modalPortNum').value),
            numberingBase
        };
        
        try {
            const response = await fetch('delete_port.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(portData)
            });
            
            const result = await response.json();
            
            if (result.success) {
                portModal.hide();
                loadOLT.call({ value: currentSite });
            } else {
                alert(result.error || 'Failed to delete port');
            }
        } catch (error) {
            console.error('Error deleting port:', error);
        }
    }
</script>
</body>
</html>