<?php
require_once 'dbconfig.php';
header('Content-Type: application/json');

try {
    // 1. Hanya izinkan POST
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        throw new Exception("Only POST requests are allowed", 405);
    }

    // 2. Ambil dan decode JSON input
    $json = file_get_contents('php://input');
    $data = json_decode($json, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        throw new Exception("Invalid JSON data", 400);
    }

    // 3. Validasi field wajib
    $required = ['siteid', 'slotid', 'port', 'fdt_code', 'update_by', 'merk'];
    foreach ($required as $field) {
        if (!isset($data[$field])) {
            throw new Exception("Missing required field: $field", 400);
        }
    }

    $siteid = $data['siteid'];
    $slotid = (int)$data['slotid'];
    $port = (int)$data['port'];
    $fdt_code = $data['fdt_code'];
    $update_by = $data['update_by'];
    $merk = $data['merk'];
    $portColumn = 'port' . $port;

    // 4. Cek nama kolom valid (harus port0 - port16)
    if ($port < 0 || $port > 16) {
        throw new Exception("Invalid port number: $port", 400);
    }

    // 5. Koneksi DB
    $db = getDBConnection();
    if (!$db) {
        throw new Exception("Database connection failed", 500);
    }
    $db->set_charset("utf8mb4");

    // 6. Jalankan UPDATE
    $sqlUpdate = "
        UPDATE port_management_olt 
        SET {$portColumn} = ?, 
            update_by = ?, 
            merk = ?, 
            updated_time = NOW() 
        WHERE siteid = ? AND slotid = ?
    ";
    $stmt = $db->prepare($sqlUpdate);
    if (!$stmt) {
        throw new Exception("Prepare failed: " . $db->error, 500);
    }

    $stmt->bind_param("ssssi", $fdt_code, $update_by, $merk, $siteid, $slotid);
    $stmt->execute();

    // 7. Jika tidak ada baris yang terupdate, lakukan INSERT
    if ($stmt->affected_rows === 0) {
        $sqlInsert = "
            INSERT INTO port_management_olt (siteid, slotid, {$portColumn}, update_by, merk, updated_time)
            VALUES (?, ?, ?, ?, ?, NOW())
        ";
        $insert = $db->prepare($sqlInsert);
        if (!$insert) {
            throw new Exception("Insert prepare failed: " . $db->error, 500);
        }

        $insert->bind_param("sisss", $siteid, $slotid, $fdt_code, $update_by, $merk);
        if (!$insert->execute()) {
            throw new Exception("Insert failed: " . $insert->error, 500);
        }

        $response = [
            'success' => true,
            'message' => 'New record inserted',
            'data' => $data
        ];
        $insert->close();
    } else {
        $response = [
            'success' => true,
            'message' => 'Port updated successfully',
            'data' => $data
        ];
    }

    $stmt->close();
    $db->close();
    
    // Clear any previous output and send JSON
    ob_clean();
    echo json_encode($response);
    exit();

} catch (Exception $e) {
    // Clear any previous output
    ob_clean();
    http_response_code($e->getCode() >= 400 ? $e->getCode() : 500);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage(),
        'debug' => [
            'timestamp' => date('c')
        ]
    ]);
    exit();
}