<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/php-error.log');

mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);

require_once 'dbconfig.php';

ob_start();

try {
    $rawSiteid = trim($_GET['siteid'] ?? '');
    if (!$rawSiteid) throw new Exception('Site ID is required');

    $db = getDBConnection();
    if (!$db) throw new Exception('Database connection failed');

    // Get brand information
    $stmt = $db->prepare("
        SELECT COALESCE(a.merk, p.merk, 'HUAWEI') AS merk,
               COALESCE(a.type_brand, '') AS type_brand
        FROM port_management_olt p
        LEFT JOIN assetdata a USING(siteid)
        WHERE p.siteid = ?
        LIMIT 1
    ");
    $stmt->bind_param('s', $rawSiteid);
    $stmt->execute();
    $row = $stmt->get_result()->fetch_assoc();
    $stmt->close();

    $merk = $row['merk'] ?? 'HUAWEI';
    $type_brand = strtoupper(trim($row['type_brand'] ?? ''));

    // Determine slot filter based on type_brand
    switch ($type_brand) {
        case 'MA5800-X15':
            $slotFilter = array_diff(range(1, 17), [8, 9]);
            break;
        case 'MA5600T':
            $slotFilter = array_diff(range(1, 16), [7, 8]);
            break;
        case 'C300':
            $slotFilter = array_diff(range(2, 17), [10, 11]);
            break;
        case '600':
            case 'C600':
            $slotFilter = array_diff(range(1, 16), [10, 11]);
            break;
        case 'MA5608T':
            $slotFilter = [0, 1];
            break;    
        case 'MA5800-X2':
        case 'C320':
        case 'C620':
            $slotFilter = [1, 2];
            break;
        case 'ZTE':
        case 'C610':
            $slotFilter = [3];
            break;
        case 'HUAWEI':
        case 'MA5801S-GP16':
        case 'MA5801S-GP16-H2':
            $slotFilter = [0];
            break;    
        case '':
            $slotFilter = range(0, 15);
            break;
        default:
            $slotFilter = range(0, 15);
    }

    // Determine port range based on brand
    $portRange = ($merk === 'ZTE') ? range(1, 16) : range(0, 15);

    // Initialize data structure
    $slotsData = [];
    $portStats = [
        'total' => 0,
        'rfs' => 0,
        'non_rfs' => 0,
        'unused' => 0,
        'type_brand' => $type_brand,
        'merk' => $merk
    ];

    // Initialize all slots and ports
    foreach ($slotFilter as $slot) {
        $slotsData[$slot] = ['slotid' => $slot];
        foreach ($portRange as $p) {
            $slotsData[$slot]["port$p"] = 'Unused';
            $slotsData[$slot]["status$p"] = 'Unused';
            $portStats['unused']++;
            $portStats['total']++;
        }
    }

    // Get actual port data
    $stmt = $db->prepare("
        SELECT *
        FROM port_management_olt
        WHERE siteid = ?
        ORDER BY slotid
    ");
    $stmt->bind_param('s', $rawSiteid);
    $stmt->execute();
    $result = $stmt->get_result();

    while ($data = $result->fetch_assoc()) {
        $slot = intval($data['slotid']);
        if (!in_array($slot, $slotFilter, true)) continue;

        foreach ($portRange as $p) {
            $pk = "port$p";
            $sk = "status$p";
            $val = trim($data[$pk] ?? '') ?: 'Unused';
            $slotsData[$slot][$pk] = $val;

            if ($val !== 'Unused') {
                // Get first 6 characters of fdt_code
                $ref = substr($val, 0, 6);
                
                // Find status in assetdata
                $ast = $db->prepare("
                    SELECT status
                    FROM assetdata
                    WHERE LEFT(siteid, 6) = ?
                    LIMIT 1
                ");
                $ast->bind_param('s', $ref);
                $ast->execute();
                $ad = $ast->get_result()->fetch_assoc();
                $ast->close();

                $stat = $ad['status'] ?? 'Unknown';
                $slotsData[$slot][$sk] = $stat;
                
                // Update statistics
                if ($stat === 'RFS') {
                    $portStats['rfs']++;
                    $portStats['unused']--;
                } elseif ($stat !== 'Unknown') {
                    $portStats['non_rfs']++;
                    $portStats['unused']--;
                }
            }
        }
    }

    ob_end_clean();
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode([
        'success' => true,
        'slots' => $slotsData,
        'stats' => $portStats
    ], JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
    exit;

} catch (Exception $ex) {
    ob_end_clean();
    http_response_code(500);
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode([
        'success' => false,
        'error' => $ex->getMessage(),
        'debug' => ['timestamp' => date('c')]
    ]);
    exit;
} finally {
    if (isset($stmt) && $stmt instanceof mysqli_stmt) $stmt->close();
    if (isset($ast) && $ast instanceof mysqli_stmt) $ast->close();
    if (isset($db) && $db instanceof mysqli) $db->close();
}
?>