<?php
require_once 'dbconfig.php';

// Start output buffering
ob_start();

try {
    // Validate input
    $rawSiteid = trim($_GET['siteid'] ?? '');
    if (empty($rawSiteid)) {
        throw new Exception("Site ID is required");
    }

    $db = getDBConnection();
    if (!$db) {
        throw new Exception("Database connection failed");
    }

    // Clean any previous output
    ob_clean();

    // Set proper JSON header
    header('Content-Type: application/json; charset=utf-8');
    
    // 1. Get OLT information
    $stmt = $db->prepare("
        SELECT DISTINCT COALESCE(a.merk, p.merk, 'HUAWEI') as merk 
        FROM port_management_olt p
        LEFT JOIN assetdata a ON p.siteid = a.siteid
        WHERE p.siteid = ?
        LIMIT 1
    ");
    
    if (!$stmt) {
        throw new Exception("Prepare failed: " . $db->error);
    }
    
    $stmt->bind_param("s", $rawSiteid);
    if (!$stmt->execute()) {
        throw new Exception("Query failed: " . $stmt->error);
    }
    
    $result = $stmt->get_result();
    $row = $result->fetch_assoc();
    $merk = $row['merk'] ?? 'HUAWEI';
    $stmt->close();

    // 2. Determine port range based on OLT type
    $portRange = ($merk === 'ZTE') ? range(1, 16) : range(0, 15);

    // 3. Get all slots for this OLT
    $stmt = $db->prepare("
        SELECT p.*, a.merk as asset_merk 
        FROM port_management_olt p
        LEFT JOIN assetdata a ON p.siteid = a.siteid
        WHERE p.siteid = ?
        ORDER BY slotid
    ");
    
    if (!$stmt) {
        throw new Exception("Prepare failed: " . $db->error);
    }
    
    $stmt->bind_param("s", $rawSiteid);
    if (!$stmt->execute()) {
        throw new Exception("Query failed: " . $stmt->error);
    }
    
    $result = $stmt->get_result();
    $slotsData = [];
    $portStats = [
        'total' => 0,
        'rfs' => 0,
        'non_rfs' => 0,
        'unused' => 0,
        'merk' => $merk,
        'merk_source' => isset($row['merk']) ? 'assetdata' : 'port_management_olt'
    ];

    // 4. Process each slot
    while ($row = $result->fetch_assoc()) {
        $slotId = $row['slotid'];
        $processedRow = ['slotid' => $slotId];
        
        // Update merk if available from assetdata
        if (!empty($row['asset_merk'])) {
            $merk = $row['asset_merk'];
            $portStats['merk'] = $merk;
            $portStats['merk_source'] = 'assetdata';
            $portRange = ($merk === 'ZTE') ? range(1, 16) : range(0, 15);
        }
        
        foreach ($portRange as $portNum) {
            $portKey = "port$portNum";
            $statusKey = "status$portNum";
            
            // Handle port value (ensure empty values become 'Unused')
            $portValue = trim($row[$portKey] ?? '');
            $portValue = empty($portValue) ? 'Unused' : $portValue;
            
            $portStats['total']++;
            $processedRow[$portKey] = $portValue;
            
            if ($portValue !== 'Unused') {
                $refSiteId = substr($portValue, 0, 9);
                $assetStmt = $db->prepare("
                    SELECT status, merk 
                    FROM assetdata 
                    WHERE siteid = ? OR siteid LIKE CONCAT(?, '%')
                    LIMIT 1
                ");
                
                if ($assetStmt && $assetStmt->bind_param("ss", $refSiteId, $refSiteId) && $assetStmt->execute()) {
                    $assetData = $assetStmt->get_result()->fetch_assoc();
                    $status = !empty($assetData['status']) ? $assetData['status'] : 'Unknown';
                    $portStats[$status === 'RFS' ? 'rfs' : 'non_rfs']++;
                    $processedRow[$statusKey] = $status;
                    
                    if (!empty($assetData['merk'])) {
                        $processedRow['port_merk'] = $assetData['merk'];
                    }
                    
                    $assetStmt->close();
                }
            } else {
                $processedRow[$statusKey] = 'Unused';
                $portStats['unused']++;
            }
        }
        
        $slotsData[$slotId] = $processedRow;
    }

    // 5. Return response
    ob_clean();
    echo json_encode([
        'success' => true,
        'slots' => $slotsData,
        'stats' => $portStats
    ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);

} catch (Exception $e) {
    ob_clean();
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage(),
        'debug' => [
            'timestamp' => date('c')
        ]
    ]);
} finally {
    if (isset($stmt)) $stmt->close();
    if (isset($assetStmt)) $assetStmt->close();
    if (isset($db)) $db->close();
    ob_end_flush();
}
?>