<?php
session_start();
header('Content-Type: application/json');

// Enable strict error reporting
error_reporting(E_ALL);
ini_set('display_errors', 0); // Don't display errors to users
ini_set('log_errors', 1);

try {
    // Validate session
    if (!isset($_SESSION['user'])) {
        throw new Exception('Session expired. Please login again.', 401);
    }

    // Get and validate input
    $input = file_get_contents('php://input');
    if (empty($input)) {
        throw new Exception('No data received', 400);
    }

    $data = json_decode($input, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        throw new Exception('Invalid JSON data', 400);
    }

    // Validate required fields
    $required = ['siteid', 'old_slot', 'new_slot'];
    foreach ($required as $field) {
        if (!isset($data[$field]) {
            throw new Exception("Missing required field: $field", 400);
        }
    }

    // Database operations
    $db = getDBConnection();
    if ($db->connect_error) {
        throw new Exception('Database connection failed', 500);
    }

    // Check if new slot exists
    $checkQuery = "SELECT COUNT(*) FROM olt_slots WHERE siteid = ? AND slotid = ?";
    $stmt = $db->prepare($checkQuery);
    if (!$stmt) {
        throw new Exception('Database prepare failed: ' . $db->error, 500);
    }

    $stmt->bind_param("si", $data['siteid'], $data['new_slot']);
    if (!$stmt->execute()) {
        throw new Exception('Slot check failed: ' . $stmt->error, 500);
    }

    $stmt->bind_result($count);
    $stmt->fetch();
    $stmt->close();

    if ($count > 0) {
        throw new Exception('Slot ID already exists', 409);
    }

    // Transaction
    $db->autocommit(false);

    try {
        // Update ports
        $portQuery = "UPDATE olt_ports SET slotid = ? WHERE siteid = ? AND slotid = ?";
        $stmt = $db->prepare($portQuery);
        if (!$stmt) {
            throw new Exception('Port update prepare failed: ' . $db->error, 500);
        }

        $stmt->bind_param("isi", $data['new_slot'], $data['siteid'], $data['old_slot']);
        if (!$stmt->execute()) {
            throw new Exception('Port update failed: ' . $stmt->error, 500);
        }
        $stmt->close();

        // Update slot
        $slotQuery = "UPDATE olt_slots SET slotid = ? WHERE siteid = ? AND slotid = ?";
        $stmt = $db->prepare($slotQuery);
        if (!$stmt) {
            throw new Exception('Slot update prepare failed: ' . $db->error, 500);
        }

        $stmt->bind_param("isi", $data['new_slot'], $data['siteid'], $data['old_slot']);
        if (!$stmt->execute()) {
            throw new Exception('Slot update failed: ' . $stmt->error, 500);
        }
        $stmt->close();

        $db->commit();
        echo json_encode(['success' => true, 'message' => 'Slot updated successfully']);

    } catch (Exception $e) {
        $db->rollback();
        throw $e;
    }

} catch (Exception $e) {
    http_response_code($e->getCode() ?: 500);
    echo json_encode([
        'error' => $e->getMessage(),
        'code' => $e->getCode()
    ]);
    exit;
}