<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);

session_start();
require_once 'dbconfig.php';

header('Content-Type: application/json');

// 1. Validasi Method Request
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'error' => 'Only POST requests are allowed']);
    exit;
}

// 2. Validasi Session
if (!isset($_SESSION['user'])) {
    http_response_code(401);
    echo json_encode(['success' => false, 'error' => 'Unauthorized access']);
    exit;
}

try {
    // 3. Parse Input Data
    $input = json_decode(file_get_contents('php://input'), true);
    if ($input === null) {
        throw new Exception('Invalid JSON input: ' . json_last_error_msg());
    }

    // 4. Validasi Field Wajib
    $required = ['siteid', 'slotid', 'port', 'merk'];
    foreach ($required as $field) {
        if (!isset($input[$field]) || $input[$field] === '') {
            throw new Exception("Missing required field: $field");
        }
    }

    // 5. Persiapan Data
    $db = getDBConnection();
    $siteid = $db->real_escape_string($input['siteid']);
    $slotid = intval($input['slotid']);
    $port = intval($input['port']);
    $merk = strtoupper($db->real_escape_string($input['merk']));
    $update_by = $db->real_escape_string($_SESSION['user']);

    // 6. Identifikasi Merk dan Validasi Port
    $port_schema = '';
    if ($merk === 'HUAWEI') {
        // Huawei menggunakan port0-port15
        if ($port < 0 || $port > 15) {
            throw new Exception("Invalid port number for Huawei. Valid range: 0-15");
        }
        $port_schema = '0-15';
    } elseif ($merk === 'ZTE') {
        // ZTE menggunakan port1-port16
        if ($port < 1 || $port > 16) {
            throw new Exception("Invalid port number for ZTE. Valid range: 1-16");
        }
        $port_schema = '1-16';
    } else {
        throw new Exception("Unsupported OLT merk: $merk");
    }

    // 7. Generate Column Names
    $portColumn = ($merk === 'ZTE') ? "port$port" : "port$port";
    $statusColumn = ($merk === 'ZTE') ? "status$port" : "status$port";

    // 8. Eksekusi Query
    $query = "UPDATE olt_slots 
              SET $portColumn = 'Unused', 
                  $statusColumn = 'Unused',
                  updated_time = NOW(),
                  update_by = ?
              WHERE siteid = ? AND slotid = ?";
    
    $stmt = $db->prepare($query);
    if (!$stmt) {
        throw new Exception("Prepare failed: " . $db->error);
    }

    $stmt->bind_param('ssi', $update_by, $siteid, $slotid);
    
    if (!$stmt->execute()) {
        throw new Exception("Execute failed: " . $stmt->error);
    }

    // 9. Handle Response
    if ($stmt->affected_rows === 0) {
        throw new Exception("No matching port found or port already unused");
    }

    echo json_encode([
        'success' => true,
        'message' => "Port $port ($merk) reset to Unused successfully",
        'merk' => $merk,
        'port_schema' => $port_schema,
        'affected_rows' => $stmt->affected_rows
    ]);

} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage(),
        'received_data' => $input ?? null
    ]);
} finally {
    if (isset($stmt)) $stmt->close();
    if (isset($db)) $db->close();
}
?>