<?php
// Enable strict error reporting for development
declare(strict_types=1);
ini_set('display_errors', '1');
ini_set('display_startup_errors', '1');
error_reporting(E_ALL);

// Start output buffering
ob_start();

// Set headers for JSON response and CORS
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With');

// Database configuration
define('DB_HOST', 'localhost');
define('DB_USER', 'sphnhdco_netadata');
define('DB_PASS', 'Shafiqa2425');
define('DB_NAME', 'sphnhdco_netadata');

/**
 * Send JSON response and terminate script
 */
function sendResponse(array $data, int $statusCode = 200): void {
    http_response_code($statusCode);
    echo json_encode($data);
    exit;
}

/**
 * Get database connection
 */
function getDBConnection(): mysqli {
    static $conn = null;
    
    if ($conn === null) {
        $conn = new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);
        
        if ($conn->connect_errno) {
            sendResponse([
                'status' => 'error',
                'message' => 'Database connection failed',
                'error' => $conn->connect_error
            ], 500);
        }
        
        $conn->set_charset('utf8mb4');
    }
    
    return $conn;
}

/**
 * Handle OPTIONS request for CORS preflight
 */
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    sendResponse(['status' => 'success']);
}

try {
    $conn = getDBConnection();
    $requestMethod = $_SERVER['REQUEST_METHOD'];
    $input = json_decode(file_get_contents('php://input'), true) ?? [];
    $queryParams = $_GET;

    // Route the request
    switch ($requestMethod) {
        case 'GET':
            if (isset($queryParams['id'])) {
                // Get single asset by ID
                $id = filter_var($queryParams['id'], FILTER_VALIDATE_INT);
                if ($id === false || $id <= 0) {
                    sendResponse([
                        'status' => 'error',
                        'message' => 'Invalid asset ID'
                    ], 400);
                }

                $stmt = $conn->prepare("SELECT * FROM assetdata WHERE id = ?");
                $stmt->bind_param('i', $id);
                $stmt->execute();
                $result = $stmt->get_result();

                if ($result->num_rows === 0) {
                    sendResponse([
                        'status' => 'error',
                        'message' => 'Asset not found'
                    ], 404);
                }

                sendResponse([
                    'status' => 'success',
                    'data' => $result->fetch_assoc()
                ]);
            } else {
                // Get all assets
                $result = $conn->query("SELECT * FROM assetdata");
                $assets = [];

                while ($row = $result->fetch_assoc()) {
                    $assets[] = $row;
                }

                sendResponse([
                    'status' => 'success',
                    'data' => $assets,
                    'count' => count($assets)
                ]);
            }
            break;

        case 'POST':
            // Validate required fields
            $required = ['kota', 'siteid', 'name', 'lat', 'lng', 'jenis', 'status'];
            $missing = array_diff($required, array_keys($input));
            
            if (!empty($missing)) {
                sendResponse([
                    'status' => 'error',
                    'message' => 'Missing required fields',
                    'missing_fields' => array_values($missing)
                ], 400);
            }

            // Create new asset
            $stmt = $conn->prepare("
                INSERT INTO assetdata 
                (kota, siteid, name, lat, lng, jenis, status) 
                VALUES (?, ?, ?, ?, ?, ?, ?)
            ");
            
            $stmt->bind_param(
                'sssssss',
                $input['kota'],
                $input['siteid'],
                $input['name'],
                $input['lat'],
                $input['lng'],
                $input['jenis'],
                $input['status']
            );

            if (!$stmt->execute()) {
                sendResponse([
                    'status' => 'error',
                    'message' => 'Failed to create asset',
                    'error' => $stmt->error
                ], 500);
            }

            $newId = $conn->insert_id;
            $stmt->close();

            // Return the newly created asset
            $stmt = $conn->prepare("SELECT * FROM assetdata WHERE id = ?");
            $stmt->bind_param('i', $newId);
            $stmt->execute();
            $result = $stmt->get_result();

            sendResponse([
                'status' => 'success',
                'message' => 'Asset created successfully',
                'data' => $result->fetch_assoc()
            ], 201);
            break;

        case 'PUT':
            // Validate required fields
            if (!isset($input['id'])) {
                sendResponse([
                    'status' => 'error',
                    'message' => 'Asset ID is required for update'
                ], 400);
            }

            $id = filter_var($input['id'], FILTER_VALIDATE_INT);
            if ($id === false || $id <= 0) {
                sendResponse([
                    'status' => 'error',
                    'message' => 'Invalid asset ID'
                ], 400);
            }

            // Check if asset exists
            $stmt = $conn->prepare("SELECT id FROM assetdata WHERE id = ?");
            $stmt->bind_param('i', $id);
            $stmt->execute();
            
            if ($stmt->get_result()->num_rows === 0) {
                sendResponse([
                    'status' => 'error',
                    'message' => 'Asset not found'
                ], 404);
            }

            // Build dynamic update query
            $fields = [];
            $params = [];
            $types = '';
            
            $fieldMap = [
                'kota' => 's',
                'siteid' => 's',
                'name' => 's',
                'lat' => 's',
                'lng' => 's',
                'jenis' => 's',
                'status' => 's'
            ];

            foreach ($fieldMap as $field => $type) {
                if (isset($input[$field])) {
                    $fields[] = "$field = ?";
                    $params[] = $input[$field];
                    $types .= $type;
                }
            }

            if (empty($fields)) {
                sendResponse([
                    'status' => 'error',
                    'message' => 'No fields to update'
                ], 400);
            }

            $types .= 'i'; // For ID parameter
            $params[] = $id;

            $query = "UPDATE assetdata SET " . implode(', ', $fields) . " WHERE id = ?";
            $stmt = $conn->prepare($query);
            $stmt->bind_param($types, ...$params);

            if (!$stmt->execute()) {
                sendResponse([
                    'status' => 'error',
                    'message' => 'Failed to update asset',
                    'error' => $stmt->error
                ], 500);
            }

            // Return the updated asset
            $stmt = $conn->prepare("SELECT * FROM assetdata WHERE id = ?");
            $stmt->bind_param('i', $id);
            $stmt->execute();
            $result = $stmt->get_result();

            sendResponse([
                'status' => 'success',
                'message' => 'Asset updated successfully',
                'data' => $result->fetch_assoc()
            ]);
            break;

        case 'DELETE':
            if (!isset($queryParams['id'])) {
                sendResponse([
                    'status' => 'error',
                    'message' => 'Asset ID is required'
                ], 400);
            }

            $id = filter_var($queryParams['id'], FILTER_VALIDATE_INT);
            if ($id === false || $id <= 0) {
                sendResponse([
                    'status' => 'error',
                    'message' => 'Invalid asset ID'
                ], 400);
            }

            // Check if asset exists
            $stmt = $conn->prepare("SELECT id FROM assetdata WHERE id = ?");
            $stmt->bind_param('i', $id);
            $stmt->execute();
            
            if ($stmt->get_result()->num_rows === 0) {
                sendResponse([
                    'status' => 'error',
                    'message' => 'Asset not found'
                ], 404);
            }

            // Delete the asset
            $stmt = $conn->prepare("DELETE FROM assetdata WHERE id = ?");
            $stmt->bind_param('i', $id);

            if (!$stmt->execute()) {
                sendResponse([
                    'status' => 'error',
                    'message' => 'Failed to delete asset',
                    'error' => $stmt->error
                ], 500);
            }

            sendResponse([
                'status' => 'success',
                'message' => 'Asset deleted successfully',
                'deleted_id' => $id
            ]);
            break;

        default:
            sendResponse([
                'status' => 'error',
                'message' => 'Method not allowed'
            ], 405);
    }
} catch (Throwable $e) {
    // Log the error
    error_log('API Error: ' . $e->getMessage() . "\n" . $e->getTraceAsString());
    
    sendResponse([
        'status' => 'error',
        'message' => 'Internal server error',
        'error' => $e->getMessage()
    ], 500);
} finally {
    // Clean output buffer and close connection if exists
    ob_end_flush();
    if (isset($conn) && $conn instanceof mysqli) {
        $conn->close();
    }
}