<?php
session_start();
if (!isset($_SESSION['user'])) {
    header("Location: login.php");
    exit;
}
require_once 'dbcrud.php';

// Handle CRUD operations
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    $db = getDBConnection();
    
    try {
        switch ($action) {
            case 'create':
                $stmt = $db->prepare("INSERT INTO assetdata (siteid, name, lat, lng, kota) VALUES (?, ?, ?, ?, ?)");
                $stmt->bind_param("ssdds", $_POST['siteid'], $_POST['name'], $_POST['lat'], $_POST['lng'], $_POST['kota']);
                $stmt->execute();
                echo json_encode(['success' => true, 'message' => 'Record created successfully']);
                break;
                
            case 'update':
                $stmt = $db->prepare("UPDATE assetdata SET name=?, lat=?, lng=?, kota=? WHERE siteid=?");
                $stmt->bind_param("sdds", $_POST['name'], $_POST['lat'], $_POST['lng'], $_POST['kota'], $_POST['siteid']);
                $stmt->execute();
                echo json_encode(['success' => true, 'message' => 'Record updated successfully']);
                break;
                
            case 'delete':
                $stmt = $db->prepare("DELETE FROM assetdata WHERE siteid=?");
                $stmt->bind_param("s", $_POST['siteid']);
                $stmt->execute();
                echo json_encode(['success' => true, 'message' => 'Record deleted successfully']);
                break;
                
            default:
                echo json_encode(['success' => false, 'message' => 'Invalid action']);
        }
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
    exit;
}

// Handle data fetch for DataTable
if (isset($_GET['ajax'])) {
    $db = getDBConnection();
    $columns = ['siteid', 'name', 'lat', 'lng', 'kota'];
    $searchValue = $_GET['search']['value'] ?? '';
    
    // Build query
    $query = "SELECT * FROM assetdata WHERE 1=1";
    if (!empty($searchValue)) {
        $query .= " AND (siteid LIKE '%$searchValue%' OR name LIKE '%$searchValue%' OR kota LIKE '%$searchValue%')";
    }
    
    // Total records
    $totalRecords = $db->query("SELECT COUNT(*) FROM assetdata")->fetch_row()[0];
    $totalFiltered = $db->query(str_replace('*', 'COUNT(*)', $query))->fetch_row()[0];
    
    // Ordering
    if (isset($_GET['order'])) {
        $orderColumn = $columns[$_GET['order'][0]['column']];
        $orderDir = $_GET['order'][0]['dir'];
        $query .= " ORDER BY $orderColumn $orderDir";
    }
    
    // Pagination
    $query .= " LIMIT " . $_GET['start'] . "," . $_GET['length'];
    
    $result = $db->query($query);
    $data = [];
    while ($row = $result->fetch_assoc()) {
        $data[] = [
            'siteid' => $row['siteid'],
            'name' => $row['name'],
            'lat' => $row['lat'],
            'lng' => $row['lng'],
            'kota' => $row['kota'],
            'actions' => ''
        ];
    }
    
    echo json_encode([
        'draw' => intval($_GET['draw']),
        'recordsTotal' => $totalRecords,
        'recordsFiltered' => $totalFiltered,
        'data' => $data
    ]);
    exit;
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Asset Data CRUD</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.datatables.net/1.11.5/css/dataTables.bootstrap5.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet">
    <style>
        .dataTables_wrapper .dataTables_filter input {
            margin-left: 0.5em;
        }
    </style>
</head>
<body>
    <?php include 'navbar.php'; ?>
    <div class="container-fluid py-3">
        <div class="card">
            <div class="card-header">
                <h5 class="card-title">Asset Data Management</h5>
                <button class="btn btn-primary btn-sm float-end" id="addAssetBtn">Add New Asset</button>
            </div>
            <div class="card-body">
                <table id="assetTable" class="table table-striped" style="width:100%">
                    <thead>
                        <tr>
                            <th>Site ID</th>
                            <th>Name</th>
                            <th>Latitude</th>
                            <th>Longitude</th>
                            <th>City</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                </table>
            </div>
        </div>
    </div>

    <!-- Asset Modal -->
    <div class="modal fade" id="assetModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="modalTitle">Asset Details</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <form id="assetForm">
                        <input type="hidden" id="action" name="action" value="create">
                        <input type="hidden" id="originalSiteId" name="originalSiteId">
                        <div class="mb-3">
                            <label for="siteid" class="form-label">Site ID</label>
                            <input type="text" class="form-control" id="siteid" name="siteid" required>
                        </div>
                        <div class="mb-3">
                            <label for="name" class="form-label">Name</label>
                            <input type="text" class="form-control" id="name" name="name" required>
                        </div>
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label for="lat" class="form-label">Latitude</label>
                                <input type="number" step="0.000001" class="form-control" id="lat" name="lat" required>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label for="lng" class="form-label">Longitude</label>
                                <input type="number" step="0.000001" class="form-control" id="lng" name="lng" required>
                            </div>
                        </div>
                        <div class="mb-3">
                            <label for="kota" class="form-label">City</label>
                            <select class="form-select select2" id="kota" name="kota" required>
                                <option value="">Select City</option>
                                <?php
                                $db = getDBConnection();
                                $cities = $db->query("SELECT DISTINCT kota FROM assetdata ORDER BY kota");
                                while ($city = $cities->fetch_assoc()) {
                                    echo '<option value="'.htmlspecialchars($city['kota']).'">'.htmlspecialchars($city['kota']).'</option>';
                                }
                                $db->close();
                                ?>
                            </select>
                        </div>
                    </form>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                    <button type="button" class="btn btn-primary" id="saveAssetBtn">Save</button>
                </div>
            </div>
        </div>
    </div>

    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.datatables.net/1.11.5/js/jquery.dataTables.min.js"></script>
    <script src="https://cdn.datatables.net/1.11.5/js/dataTables.bootstrap5.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>
    <script>
    $(document).ready(function() {
        // Initialize Select2
        $('.select2').select2({
            width: '100%',
            dropdownParent: $('#assetModal')
        });
        
        // Initialize DataTable
        const table = $('#assetTable').DataTable({
            processing: true,
            serverSide: true,
            ajax: {
                url: 'crud.php?ajax=1',
                type: 'GET'
            },
            columns: [
    { data: 'siteid' },
    { data: 'name' },
    { data: 'lat' },
    { data: 'lng' },
    { data: 'kota' },
    { 
        data: null,
        render: function(data, type, row) {
            return '<button class="btn btn-sm btn-warning edit-btn" data-id="' + row.siteid + '">Edit</button> ' +
                   '<button class="btn btn-sm btn-danger delete-btn" data-id="' + row.siteid + '">Delete</button>';
        },
        orderable: false,
        searchable: false
    }
]
        });
        
        // Add new asset
        $('#addAssetBtn').click(function() {
            $('#assetForm')[0].reset();
            $('#action').val('create');
            $('#originalSiteId').val('');
            $('#modalTitle').text('Add New Asset');
            $('#assetModal').modal('show');
        });
        
        // Edit asset
        $(document).on('click', '.edit-btn', function() {
            const siteid = $(this).data('id');
            $.get('crud.php', { get_asset: 1, siteid: siteid }, function(data) {
                $('#siteid').val(data.siteid);
                $('#originalSiteId').val(data.siteid);
                $('#name').val(data.name);
                $('#lat').val(data.lat);
                $('#lng').val(data.lng);
                $('#kota').val(data.kota).trigger('change');
                $('#action').val('update');
                $('#modalTitle').text('Edit Asset');
                $('#assetModal').modal('show');
            }, 'json');
        });
        
        // Delete asset
        $(document).on('click', '.delete-btn', function() {
            if (confirm('Are you sure you want to delete this asset?')) {
                const siteid = $(this).data('id');
                $.post('crud.php', {
                    action: 'delete',
                    siteid: siteid
                }, function(response) {
                    if (response.success) {
                        table.ajax.reload();
                        alert(response.message);
                    } else {
                        alert('Error: ' + response.message);
                    }
                }, 'json');
            }
        });
        
        // Save asset
        $('#saveAssetBtn').click(function() {
            const formData = $('#assetForm').serialize();
            $.post('crud.php', formData, function(response) {
                if (response.success) {
                    $('#assetModal').modal('hide');
                    table.ajax.reload();
                    alert(response.message);
                } else {
                    alert('Error: ' + response.message);
                }
            }, 'json');
        });
        
        // Get single asset data (for edit)
        if (isset($_GET['get_asset'])) {
            $siteid = $_GET['siteid'];
            $db = getDBConnection();
            $stmt = $db->prepare("SELECT * FROM assetdata WHERE siteid = ?");
            $stmt->bind_param("s", $siteid);
            $stmt->execute();
            $result = $stmt->get_result();
            if ($result->num_rows > 0) {
                echo json_encode($result->fetch_assoc());
            } else {
                echo json_encode(['error' => 'Asset not found']);
            }
            exit;
        }
    });
    </script>
</body>
</html>